<?php

namespace App\Http\Controllers;

use App\Models\EmploymentHistory;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Exception;

class EmploymentHistoryController extends Controller
{
    public function index(): JsonResponse
    {
        try {
            $history = EmploymentHistory::with('user')->get();
            return response()->json(['success' => true, 'data' => $history, 'message' => 'Employment history retrieved successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to retrieve employment history: ' . $e->getMessage()], 500);
        }
    }

    public function store(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'User_id' => 'required|exists:users,User_id',
                'Company_name' => 'required|string|max:255',
                'Position' => 'required|string|max:255',
                'Start_date' => 'required|date',
                'End_date' => 'nullable|date|after_or_equal:Start_date',
                'Currently_working' => 'boolean',
                'Description' => 'nullable|string|max:65535',
            ]);

            $history = EmploymentHistory::create($validated);
            $history->load('user');
            return response()->json(['success' => true, 'data' => $history, 'message' => 'Employment history created successfully'], 201);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to create employment history: ' . $e->getMessage()], 500);
        }
    }

    public function show($id): JsonResponse
    {
        try {
            $history = EmploymentHistory::with('user')->findOrFail($id);
            return response()->json(['success' => true, 'data' => $history, 'message' => 'Employment history retrieved successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Employment history not found: ' . $e->getMessage()], 404);
        }
    }

    public function update(Request $request, $id): JsonResponse
    {
        try {
            $history = EmploymentHistory::findOrFail($id);
            $validated = $request->validate([
                'User_id' => 'required|exists:users,User_id',
                'Company_name' => 'required|string|max:255',
                'Position' => 'required|string|max:255',
                'Start_date' => 'required|date',
                'End_date' => 'nullable|date|after_or_equal:Start_date',
                'Currently_working' => 'boolean',
                'Description' => 'nullable|string|max:65535',
            ]);

            $history->update($validated);
            $history->load('user');
            return response()->json(['success' => true, 'data' => $history, 'message' => 'Employment history updated successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to update employment history: ' . $e->getMessage()], 500);
        }
    }

    public function destroy($id): JsonResponse
    {
        try {
            $history = EmploymentHistory::findOrFail($id);
            $history->delete();
            return response()->json(['success' => true, 'data' => null, 'message' => 'Employment history deleted successfully'], 204);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to delete employment history: ' . $e->getMessage()], 500);
        }
    }
}
